//
// Collision simulator
// ===================
//
// Filename: collision_sim.c
// Version:  1.0
// Author:   PH
// Modified:
// Date:     07/08/2005 
//
// Copyright 2005 by Out of the Hat Solutions plc <outofthehat@blueyonder.co.uk>
// The License.txt file describes the conditions under which this software may be distributed.
//-------------------------------------------------------------------------------------------------
// CHANGE HISTORY
// ==============
//
// Ver   Date        Author  Comments 
// 1.0   07/08/2005  PH      Initial revision
//
//-------------------------------------------------------------------------------------------------

#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>

#define MAX_VEHS 512
#define NUM_SIMS 2
#define ACCEL_G 9.8

#ifndef max
  #define max(a,b)((a) > (b)) ? (a) : (b)
#endif

#ifndef MAX_PATH
  #define MAX_PATH 255
#endif

FILE *outfile;

char inifile[MAX_PATH+1];
char outfilename[MAX_PATH+1];

double CONVTOMS;
double CONVFROMMS;

double min_event_delta_time   = 5;    // seconds
double max_event_delta_time   = 10;   // seconds
double min_vehicle_delta_time = 2;    // seconds
double max_vehicle_delta_time = 6;    // seconds
double min_event_position     = 100;  // metres
double max_event_position     = 1100; // metres
int max_events                = 1000;
double vehicle_speed[NUM_SIMS];
double min_reaction_time[NUM_SIMS];
double max_reaction_time[NUM_SIMS];
double min_vehicle_deceleration[NUM_SIMS];
double max_vehicle_deceleration[NUM_SIMS];
char separator;
int print_event_number;
int print_event_time;
int print_event_position;
int print_vehicle_number;
int print_vehicle_speed;
int print_vehicle_entry_time;
int print_vehicle_position;
int print_distance_to;
int print_vehicle_deceleration;
int print_reaction_time;
int print_reaction_distance;
int print_braking_distance;
int print_stopping_distance;
int print_braked_distance;
int print_collision_speed;
int print_time_to_at_speed;
int print_time_to_collision;

typedef struct
{
  int vehicle_number;
  double vehicle_position;
  double entry_time;
}VEH_REC;

VEH_REC vehicle_record[NUM_SIMS][MAX_VEHS];
int num_vehicles_stored[NUM_SIMS];
int num_collisions[NUM_SIMS];

double event_delta_time;
double event_position;
double vehicle_delta_time;
double cumulative_time;
int events;
int vehicles;    
int nearest_vehicle_to_event_pos[NUM_SIMS];

//-------------------------------------------------------------------------------------------------
// Functions for reading ini file
//-------------------------------------------------------------------------------------------------

double getprofiledbl(char *section, char *key)
{
  char temp[101];
  
  if (GetPrivateProfileString(section, key, "", temp, 100, inifile) == 0)
  {
    printf("INI FILE ERROR: section: %s key: %s\n",section, key);
    exit(1);
  }
  return atof(temp);
}

void getprofilestr(char *section, char *key, char *val, char len)
{
  
  if (GetPrivateProfileString(section, key, "", val, len, inifile) == 0)
  {
    printf("INI FILE ERROR: section: %s key: %s\n",section, key);
    exit(1);
  }
}

int getprofilebool(char *section, char *key)
{
  char temp[101];
  
  if (GetPrivateProfileString(section, key, "", temp, 100, inifile) == 0)
  {
    printf("INI FILE ERROR: section: %s key: %s\n",section, key);
    exit(1);
  }
  if (temp[0] == 'Y' || temp[0] == 'y')
  {
    return 1;
  }
  else
  {
    return 0;
  }
}

//-------------------------------------------------------------------------------------------------
// Functions for data output
//-------------------------------------------------------------------------------------------------

void print_newline(void)
{
  if (outfile != NULL)
  {
    fprintf(outfile, "\n");
  }
  else
  {
    printf("\n");
  }
}

void print_sep(void)
{
  if (outfile != NULL)
  {
    fprintf(outfile, "%c", separator);
  }
  else
  {
    printf("%c", separator);
  }
}

void print_head( int enabled, char *head)
{
  if (enabled)
  {
    print_sep();
    if (outfile != NULL)
    {
      fprintf(outfile, head);
    }
    else
    {
      printf(head);
    }
  }
}

void print_var( int enabled, char *format, double var)
{
  if (enabled)
  {
    print_sep();
    if (outfile != NULL)
    {
      fprintf(outfile, format, var);
    }
    else
    {
      printf(format, var);
    }
  }
}

//-------------------------------------------------------------------------------------------------
// Functions required for simulator
//-------------------------------------------------------------------------------------------------

double random_number(double minf, double maxf)
{
  if (maxf == minf)
  {
    return (maxf);
  }
  return (((double)rand() * (maxf - minf) / RAND_MAX) + minf);
}

//-------------------------------------------------------------------------------------------------
// Initialisation
//-------------------------------------------------------------------------------------------------

void init_sim(void)
{
  int n,s;
  char temp[101];

  // Read parameters from ini file

  getprofilestr("Settings", "Speed units", temp, 100);
  
  if (temp[0] == 'M' || temp[0] == 'm')
  {
    CONVTOMS = (double)40 / 90;
    CONVFROMMS = (double)90 / 40;
  }
  else if (temp[0] == 'K' || temp[0] == 'k')
  {
    CONVTOMS = (double)10 / 36;
    CONVFROMMS = (double)36 / 10;
  }
  else
  {
    printf("INI FILE ERROR: Speed units not MPH or KMH\n");
    exit(1);
  }

  max_events = (int)getprofiledbl("Settings", "Number of events");
  min_event_delta_time   = getprofiledbl("Settings", "Minimum time between events");
  max_event_delta_time   = getprofiledbl("Settings", "Maximum time between events");
  if (min_event_delta_time > max_event_delta_time)
  {
    printf("INI FILE ERROR: Minimum time between events greater than max\n");
    exit(1);
  }
    
  min_vehicle_delta_time   = getprofiledbl("Settings", "Minimum time between vehicles");
  max_vehicle_delta_time   = getprofiledbl("Settings", "Maximum time between vehicles");
  if (min_vehicle_delta_time > max_vehicle_delta_time)
  {
    printf("INI FILE ERROR: Minimum time between vehicles greater than max\n");
    exit(1);
  }
    
  min_event_position   = getprofiledbl("Settings", "Minimum event position");
  max_event_position   = getprofiledbl("Settings", "Maximum event position");
  if (min_event_position > max_event_position)
  {
    printf("INI FILE ERROR: Minimum event position greater than max\n");
    exit(1);
  }
    
  for(s = 0; s < NUM_SIMS; s++)
  {
    sprintf(temp, "Simulation %d", s + 1);

    vehicle_speed[s] = getprofiledbl(temp, "Vehicle speed") * CONVTOMS;
    min_reaction_time[s] = getprofiledbl(temp, "Minimum reaction time");
    max_reaction_time[s] = getprofiledbl(temp, "Maximum reaction time");
    if (min_reaction_time[s] > max_reaction_time[s])
    {
      printf("INI FILE ERROR: Minimum reaction time greater than max\n");
      exit(1);
    }
    min_vehicle_deceleration[s] = getprofiledbl(temp, "Minimum deceleration");
    max_vehicle_deceleration[s] = getprofiledbl(temp, "Maximum deceleration");
    if (min_vehicle_deceleration[s] > max_vehicle_deceleration[s])
    {
      printf("INI FILE ERROR: Minimum deceleration greater than max\n");
      exit(1);
    }
  }

  GetPrivateProfileString("Output", "Filename", "", outfilename, MAX_PATH, inifile);
  
  getprofilestr("Output", "Separator", temp, 100);
  
  if (temp[0] == 'C' || temp[0] == 'c')
  {
    separator = ',';
  }
  else if (temp[0] == 'T' || temp[0] == 't')
  {
    separator = '\t';
  }
  else
  {
    printf("INI FILE ERROR: Separator not TAB or COMMA\n");
    exit(1);
  }

  print_event_number         = getprofilebool("Output", "Print Event number");
  print_event_time           = getprofilebool("Output", "Print Event time");
  print_event_position       = getprofilebool("Output", "Print Event position");
  print_vehicle_number       = getprofilebool("Output", "Print Vehicle number");
  print_vehicle_speed        = getprofilebool("Output", "Print Vehicle speed");
  print_vehicle_entry_time   = getprofilebool("Output", "Print Vehicle entry time");
  print_vehicle_position     = getprofilebool("Output", "Print Vehicle position");
  print_distance_to          = getprofilebool("Output", "Print Distance to pos");
  print_vehicle_deceleration = getprofilebool("Output", "Print Vehicle deceleration");
  print_reaction_time        = getprofilebool("Output", "Print Reaction time");
  print_reaction_distance    = getprofilebool("Output", "Print Reaction distance");
  print_braking_distance     = getprofilebool("Output", "Print Braking distance");
  print_stopping_distance    = getprofilebool("Output", "Print Stopping distance");
  print_braked_distance      = getprofilebool("Output", "Print Braked distance");
  print_collision_speed      = getprofilebool("Output", "Print Collision speed");
  print_time_to_at_speed     = getprofilebool("Output", "Print Time to pos at spd");
  print_time_to_collision    = getprofilebool("Output", "Print Time to collision");
  
  // Initialise sim
  
  for(s = 0; s < NUM_SIMS; s++)
  {
    for (n = 0; n < MAX_VEHS; n++)
    {
      vehicle_record[s][n].vehicle_number = 0;
      vehicle_record[s][n].vehicle_position = 0;
      vehicle_record[s][n].entry_time = 0;
    }
    num_vehicles_stored[s] = 0;
    nearest_vehicle_to_event_pos[s] = 0;
    num_collisions[s] = 0;
  }

  events = 0;
  vehicles = 0;
  cumulative_time = 0;
  vehicle_delta_time = 0;
  event_delta_time = random_number(min_event_delta_time, max_event_delta_time);
  event_position   = random_number(min_event_position, max_event_position);
}

//-------------------------------------------------------------------------------------------------
// Simulation functions
//-------------------------------------------------------------------------------------------------

void check_and_record_collision(void)
{
  double reaction_time[NUM_SIMS];
  double vehicle_deceleration[NUM_SIMS];
  double collision_speed[NUM_SIMS];
  double reaction_distance[NUM_SIMS];
  double braking_distance[NUM_SIMS];
  double stopping_distance[NUM_SIMS];
  double braked_distance[NUM_SIMS];
  double distance_to[NUM_SIMS];
  double time_to_at_speed[NUM_SIMS];
  double time_to_collision[NUM_SIMS];
  int s;
  int collision_occurred = 0;
  
  for (s = 0; s < NUM_SIMS; s++)
  {
    reaction_time[s] = random_number(min_reaction_time[s], max_reaction_time[s]);
    vehicle_deceleration[s] = random_number(min_vehicle_deceleration[s], max_vehicle_deceleration[s]) * ACCEL_G;
    // Calculate stopping distance
    reaction_distance[s] = reaction_time[s] * vehicle_speed[s];
    braking_distance[s] = (vehicle_speed[s] * vehicle_speed[s]) / (2 * vehicle_deceleration[s]);
    stopping_distance[s] = reaction_distance[s] + braking_distance[s];

    // get distance of closest vehicle
    distance_to[s] = event_position - vehicle_record[s][nearest_vehicle_to_event_pos[s]].vehicle_position;

    // Calculate the amount of time to reach event position without slowing down
    // (not used in calculation, for recording purposes only)
    time_to_at_speed[s] = distance_to[s] / vehicle_speed[s];

    // If vehicle is within it's stopping distance of hazard, collision occurs. 
    if (distance_to[s] <= stopping_distance[s])
    {
      // Calculate the collision speed.
      // If distance is less than reaction distance then no braking
      // takes place, so collision speed = speed.
      if (distance_to[s] <= reaction_distance[s])
      {
        collision_speed[s] = vehicle_speed[s];

        // Calculate the time to collision
        // (not used in calculation, for recording purposes only)
        time_to_collision[s] = distance_to[s] / vehicle_speed[s];
        braked_distance[s] = 0;
      }
      else // braking does take place.
      {
        braked_distance[s] = distance_to[s] - reaction_distance[s];
        collision_speed[s] = sqrt((vehicle_speed[s] * vehicle_speed[s]) - (2 * vehicle_deceleration[s] * braked_distance[s]));

        // Calculate the time to collision
        // (not used in calculation, for recording purposes only)
        time_to_collision[s] = reaction_time[s] + ((vehicle_speed[s] - collision_speed[s]) / vehicle_deceleration[s]);
      }

      // Filnally, update the number of collisions
      num_collisions[s]++;
      collision_occurred = 1;
    }
    else // no collision takes place
    {
      braked_distance[s] = braking_distance[s];
      time_to_collision[s] = 999.999;  // no collision, just make this a large number.
      collision_speed[s] = 0;
    }
  }

  // Output data if collision(s) occurred
  if (collision_occurred)
  {
    print_var( print_event_number, "%8.0f", (double)events);
    print_var( print_event_time, "%8.1f", cumulative_time);
    print_var( print_event_position, "%8.1f", event_position);
    for (s = 0; s < NUM_SIMS; s++)
    {
      print_sep();
      print_var( print_vehicle_number, "%8.0f", (double)vehicle_record[s][nearest_vehicle_to_event_pos[s]].vehicle_number); 
      print_var( print_vehicle_speed, "%4.0f", vehicle_speed[s] * CONVFROMMS ); 
      print_var( print_vehicle_entry_time, "%8.1f", vehicle_record[s][nearest_vehicle_to_event_pos[s]].entry_time); 
      print_var( print_vehicle_position, "%8.1f", vehicle_record[s][nearest_vehicle_to_event_pos[s]].vehicle_position); 
      print_var( print_distance_to, "%6.1f", distance_to[s]); 
      print_var( print_vehicle_deceleration, "%6.3f", vehicle_deceleration[s] / ACCEL_G); 
      print_var( print_reaction_time, "%6.3f", reaction_time[s]); 
      print_var( print_reaction_distance, "%6.1f", reaction_distance[s]);
      print_var( print_braking_distance, "%6.1f", braking_distance[s]);
      print_var( print_stopping_distance, "%6.1f", stopping_distance[s]);
      print_var( print_braked_distance, "%6.1f", braked_distance[s]);
      print_var( print_collision_speed, "%6.1f", collision_speed[s] * CONVFROMMS );
      print_var( print_time_to_at_speed, "%6.3f", time_to_at_speed[s]);
      print_var( print_time_to_collision, "%6.3f", time_to_collision[s]);
    } 
    print_newline();
  }
}

void adjust_vehicle_pos(double delta_time, int record_nearest)
{
  int n,s;
  for (s = 0; s < NUM_SIMS; s++)
  {
    for (n = 0; n < num_vehicles_stored[s]; n++)
    {
      vehicle_record[s][n].vehicle_position += (delta_time * vehicle_speed[s]);

      // Record index of vehicle nearest to event (but not past it)
      if ((record_nearest != 0) && (vehicle_record[s][n].vehicle_position <= event_position))
      {
        nearest_vehicle_to_event_pos[s] = n;
      }

      // Lose any vehicles which have already passed max_event_position
      if (vehicle_record[s][n].vehicle_position > max_event_position)
      {
        num_vehicles_stored[s] = n;
        break;
      }
    }
  }
}

void new_event(void)
{
  adjust_vehicle_pos(event_delta_time, 1);
  check_and_record_collision();
}

void new_vehicle(void)
{
  int s;

  // First, update existing vehicle records
  adjust_vehicle_pos(vehicle_delta_time, 0);

  for (s = 0; s < NUM_SIMS; s++)
  {
    if (num_vehicles_stored[s] > 0)
    {
      // Move all vehicles up one position, memmove used for speed
      memmove(&vehicle_record[s][1], &vehicle_record[s][0], sizeof(VEH_REC) * max(num_vehicles_stored[s], (MAX_VEHS - 1)));
    }
    // Insert new vehicle at index pos 0
    vehicle_record[s][0].vehicle_number = vehicles + 1;
    vehicle_record[s][0].vehicle_position = 0;
    vehicle_record[s][0].entry_time = cumulative_time;
    
    num_vehicles_stored[s]++;
    if (num_vehicles_stored[s] > MAX_VEHS)
    {
      printf("ERROR: MAX_VEHS exceeded\n)");
      exit(1);
    }
  }
}

int main(int argc, char *argv[])
{
  int s;
  
  if (argc == 2)
  {
    strcpy (inifile, argv[1]);  
  }
  else
  {
    strcpy (inifile, ".\\collision_sim.ini");  
  }
  
  // Seed the random number generator
  srand((unsigned)time(NULL)*((unsigned)time(NULL)+1));

  // Initialise parameters
  init_sim();

  if (outfilename != "")
  {
    outfile = fopen( outfilename, "w");
    if (outfile == NULL)
    {
      printf ("ERROR: Could not open file %s\n", outfilename);
      exit(1);
    }      
  }
  else
  {
    outfile = NULL;
  }

  // Output column headers
  print_head( print_event_number, "Event");
  print_head( print_event_time, "Event");
  print_head( print_event_position, "Event");
  for(s = 0; s < NUM_SIMS; s++)
  {
    print_sep();
    print_head( print_vehicle_number, "Vehicle"); 
    print_head( print_vehicle_speed, "Vehicle"); 
    print_head( print_vehicle_entry_time, "Vehicle"); 
    print_head( print_vehicle_position, "Vehicle"); 
    print_head( print_distance_to, "Distance"); 
    print_head( print_vehicle_deceleration, "Vehicle"); 
    print_head( print_reaction_time, "Reaction"); 
    print_head( print_reaction_distance, "Reaction");
    print_head( print_braking_distance, "Braking");
    print_head( print_stopping_distance, "Stopping");
    print_head( print_braked_distance, "Braked");
    print_head( print_collision_speed, "Collision");
    print_head( print_time_to_at_speed, "Time to");
    print_head( print_time_to_collision, "Time to");
  } 
  print_newline();
  print_head( print_event_number, "number");
  print_head( print_event_time, "time");
  print_head( print_event_position, "position");
  for(s = 0; s < NUM_SIMS; s++)
  {
    print_sep();
    print_head( print_vehicle_number, "number"); 
    print_head( print_vehicle_speed, "speed"); 
    print_head( print_vehicle_entry_time, "entry time"); 
    print_head( print_vehicle_position, "position"); 
    print_head( print_distance_to, "to pos"); 
    print_head( print_vehicle_deceleration, "deceleration"); 
    print_head( print_reaction_time, "time"); 
    print_head( print_reaction_distance, "distance");
    print_head( print_braking_distance, "distance");
    print_head( print_stopping_distance, "distance");
    print_head( print_braked_distance, "distance");
    print_head( print_collision_speed, "speed");
    print_head( print_time_to_at_speed, "pos at spd");
    print_head( print_time_to_collision, "collision");
  } 
  print_newline();

  // Do the simulation
  while (events <= max_events)
  {
    while (vehicle_delta_time < event_delta_time)
    {
      new_vehicle();
      vehicles++;
      cumulative_time += vehicle_delta_time;
      event_delta_time -= vehicle_delta_time;
      vehicle_delta_time = random_number(min_vehicle_delta_time, max_vehicle_delta_time);
    }    
    while (event_delta_time < vehicle_delta_time)
    {
      new_event();
      events++;
      cumulative_time += event_delta_time;
      vehicle_delta_time -= event_delta_time;
      event_delta_time = random_number(min_event_delta_time, max_event_delta_time);
      event_position   = random_number(min_event_position, max_event_position);
    }    
  }

  //Events will now be one more than max, so adjust.
  events--;

  // Output the summary data  
  print_newline();
  printf("Events     %c%8d\n", separator, events);
  printf("Vehicles   %c%8d\n", separator, vehicles);
  printf("Speed      %c%8.0f%c%8.0f\n", separator, vehicle_speed[0] * CONVFROMMS, separator, vehicle_speed[1] * CONVFROMMS);
  printf("Collisions %c%8d%c%8d\n", separator, num_collisions[0], separator, num_collisions[1]);
  printf("Risk/Event %c%8.4f%c%8.4f\n", separator, (double)num_collisions[0] / events, separator, (double)num_collisions[1] / events);
  if (num_collisions[0] > 0 && num_collisions[1] > 0)
  {
    printf("Risk ratio %c%8.4f%c%8.4f\n", separator, (double)num_collisions[0] / num_collisions[1], separator, (double)num_collisions[1] / num_collisions[0]);
  }
  else
  {
    printf("Risk ratio cannot be calculated\n");
  }

  if (outfile != NULL)
  {
    fprintf(outfile, "Events     %c%8d\n", separator, events);
    fprintf(outfile, "Vehicles   %c%8d\n", separator, vehicles);
    fprintf(outfile, "Speed      %c%8.0f%c%8.0f\n", separator, vehicle_speed[0] * CONVFROMMS, separator, vehicle_speed[1] * CONVFROMMS);
    fprintf(outfile, "Collisions %c%8d%c%8d\n", separator, num_collisions[0], separator, num_collisions[1]);
    fprintf(outfile, "Risk/Event %c%8.4f%c%8.4f\n", separator, (double)num_collisions[0] / events, separator, (double)num_collisions[1] / events);
    if (num_collisions[0] > 0 && num_collisions[1] > 0)
    {
      fprintf(outfile, "Risk ratio %c%8.4f%c%8.4f\n", separator, (double)num_collisions[0] / num_collisions[1], separator, (double)num_collisions[1] / num_collisions[0]);
    }
    else
    {
      fprintf(outfile, "Risk ratio cannot be calculated\n");
    }
    fclose(outfile);
  }
  system("PAUSE");	
  return 0;
}
